/*
 * Decompiled with CFR 0.152.
 */
package org.esa.snap.vfs.remote.s3;

import java.net.URL;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Hex;
import org.esa.snap.vfs.preferences.model.Property;

class S3AuthenticationV4 {
    private static final DateTimeFormatter ISO_DATE_TIME = DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss'Z'");
    private static final DateTimeFormatter ISO_DATE = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final long SIGNING_KEY_VALIDITY = 14L;
    private static final String HTTP_VERB_NAME = "<HTTPMethod>";
    private static final String CANONICAL_URI_NAME = "<CanonicalURI>";
    private static final String CANONICAL_QUERY_STRING_NAME = "<CanonicalQueryString>";
    private static final String CANONICAL_HEADERS_NAME = "<CanonicalHeaders>";
    private static final String SIGNED_HEADERS_NAME = "<SignedHeaders>";
    private static final String HASHED_PAYLOAD_NAME = "<HashedPayload>";
    private static final String CANONICAL_REQUEST_HASH_NAME = "<CanonicalRequest>";
    private static final String CANONICAL_REQUEST_VALUE = "<HTTPMethod>\n<CanonicalURI>\n<CanonicalQueryString>\n<CanonicalHeaders>\n<SignedHeaders>\n<HashedPayload>";
    private static final String AWS_SIGNATURE_ALGORITHM_NAME = "<Algorithm>";
    private static final String AWS_SIGNATURE_ALGORITHM_VALUE = "AWS4-HMAC-SHA256";
    private static final String TIMESTAMP_NAME = "<TimeStamp>";
    private static final String SCOPE_NAME = "<Scope>";
    private static final String DATE_STAMP_NAME = "<DateStamp>";
    private static final String AWS_REGION_NAME = "<AWSRegion>";
    private static final String AWS_SERVICE_VALUE = "s3";
    private static final String SCOPE_VALUE = "<DateStamp>/<AWSRegion>/s3/aws4_request";
    private static final String STRING_TO_SIGN_VALUE = "<Algorithm>\n<TimeStamp>\n<Scope>\n<CanonicalRequest>";
    private static final String SIGNING_KEY_DATA_VALUE = "aws4_request";
    private static final String AWS_HOST_HEADER_NAME = "host";
    private static final String AWS_CONTENT_SHA256_HEADER_NAME = "x-amz-content-sha256";
    private static final String AWS_CONTENT_SHA256_HEADER_VALUE = "UNSIGNED-PAYLOAD";
    private static final String AWS_DATE_HEADER_NAME = "x-amz-date";
    private static final String AWS_ACCESS_KEY_ID_NAME = "<Credential>";
    private static final String AWS_SIGNATURE_NAME = "<Signature>";
    private static final String AWS_AUTHORIZATION_TOKEN_VALUE = "<Algorithm> Credential=<Credential>/<Scope>, SignedHeaders=<SignedHeaders>, Signature=<Signature>";
    private final String httpVerb;
    private final String region;
    private final String accessKeyId;
    private final String secretAccessKey;
    private final Map<String, String> customParameters;
    private LocalDateTime creationDate;
    private LocalDateTime expirationDate = null;
    private Map<String, String> awsHeaders;
    private byte[] signingKey;
    private URL lastAuthorizedURL;
    private String lastAuthorizationToken;

    S3AuthenticationV4(String httpVerb, String region, String accessKeyId, String secretAccessKey, Map<String, String> customParameters) {
        this.httpVerb = httpVerb;
        this.region = region;
        this.accessKeyId = accessKeyId;
        this.secretAccessKey = secretAccessKey;
        this.customParameters = customParameters;
        this.creationDate = LocalDateTime.now(ZoneOffset.UTC);
        this.lastAuthorizedURL = null;
        this.lastAuthorizationToken = null;
    }

    private static String lowercase(String value) {
        String lowercase = "";
        if (value != null) {
            lowercase = value.toLowerCase();
        }
        return lowercase;
    }

    private static String trim(String value) {
        String trim = "";
        if (value != null) {
            trim = value.trim();
        }
        return trim;
    }

    private static String hex(byte[] data) {
        String hex = "";
        if (data != null) {
            hex = String.valueOf(Hex.encodeHex((byte[])data));
        }
        return hex;
    }

    private static byte[] sha256Hash(byte[] data) {
        byte[] sha256Hash = new byte[]{};
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            sha256Hash = digest.digest(data);
        }
        catch (Exception exception) {
            // empty catch block
        }
        return sha256Hash;
    }

    private static byte[] hmacSHA256Hash(byte[] data, byte[] secretKey) {
        byte[] hmacSHA256Hash = new byte[]{};
        try {
            SecretKeySpec secretKeySpec = new SecretKeySpec(secretKey, "HmacSHA256");
            Mac mac = Mac.getInstance("HmacSHA256");
            mac.init(secretKeySpec);
            hmacSHA256Hash = mac.doFinal(data);
        }
        catch (Exception exception) {
            // empty catch block
        }
        return hmacSHA256Hash;
    }

    private static String uriEncode(String input, boolean encodeSlash) {
        StringBuilder result = new StringBuilder();
        try {
            for (int i = 0; i < input.length(); ++i) {
                char ch = input.charAt(i);
                if (ch >= 'A' && ch <= 'Z' || ch >= 'a' && ch <= 'z' || ch >= '0' && ch <= '9' || ch == '_' || ch == '-' || ch == '~' || ch == '.' || ch == '%') {
                    result.append(ch);
                    continue;
                }
                if (ch == '/') {
                    result.append(encodeSlash ? "%2F" : Character.valueOf(ch));
                    continue;
                }
                result.append(URLEncoder.encode(String.valueOf(ch), "UTF8"));
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return result.toString();
    }

    private static List<Property> getRequestParametersList(String query) {
        ArrayList<Property> requestParameters = new ArrayList<Property>();
        if (query != null && !((String)query).isEmpty()) {
            String[] requestParametersArray;
            query = ((String)query).endsWith("&") ? query : (String)query + "&";
            String requestParametersString = ((String)query).replaceAll("(.*?)=(.*?)&", "$1=$2\n");
            for (String requestParameter : requestParametersArray = requestParametersString.split("\n")) {
                String requestParameterName = requestParameter.replaceAll("(.*)=(.*)", "$1");
                String requestParameterValue = requestParameter.replaceAll("(.*)=(.*)", "$2");
                requestParameters.add(new Property(requestParameterName, requestParameterValue));
            }
            requestParameters.sort(new PropertyCodePointSorter());
        }
        return requestParameters;
    }

    private Map<String, String> buildAwsHeaders(URL url) {
        LinkedHashMap<String, String> headers = new LinkedHashMap<String, String>();
        headers.put(AWS_HOST_HEADER_NAME, url.getHost());
        headers.put(AWS_CONTENT_SHA256_HEADER_NAME, AWS_CONTENT_SHA256_HEADER_VALUE);
        headers.put(AWS_DATE_HEADER_NAME, this.getISODateTime());
        for (Map.Entry<String, String> customParameter : this.customParameters.entrySet()) {
            if (customParameter.getKey().contentEquals(AWS_CONTENT_SHA256_HEADER_NAME) || customParameter.getKey().contentEquals(AWS_DATE_HEADER_NAME)) continue;
            headers.put(customParameter.getKey(), customParameter.getValue());
        }
        return headers;
    }

    private String buildCanonicalQueryString(List<Property> requestParameters) {
        StringBuilder canonicalQueryString = new StringBuilder();
        boolean building = false;
        for (Property requestParameter : requestParameters) {
            if (building) {
                canonicalQueryString.append("&");
            } else {
                building = true;
            }
            canonicalQueryString.append(S3AuthenticationV4.uriEncode(requestParameter.getName(), true));
            canonicalQueryString.append("=");
            canonicalQueryString.append(S3AuthenticationV4.uriEncode(requestParameter.getValue(), true));
        }
        return canonicalQueryString.toString();
    }

    private String buildCanonicalHeaders() {
        StringBuilder canonicalHeaders = new StringBuilder();
        Set<Map.Entry<String, String>> awsHeadersSet = this.awsHeaders.entrySet();
        for (Map.Entry<String, String> awsHeader : awsHeadersSet) {
            canonicalHeaders.append(S3AuthenticationV4.lowercase(awsHeader.getKey()));
            canonicalHeaders.append(':');
            canonicalHeaders.append(S3AuthenticationV4.trim(awsHeader.getValue()));
            canonicalHeaders.append('\n');
        }
        return canonicalHeaders.toString();
    }

    private String buildSignedHeaders() {
        StringBuilder signedHeaders = new StringBuilder();
        Set<Map.Entry<String, String>> awsHeadersSet = this.awsHeaders.entrySet();
        boolean building = false;
        for (Map.Entry<String, String> awsHeader : awsHeadersSet) {
            if (building) {
                signedHeaders.append(";");
            } else {
                building = true;
            }
            signedHeaders.append(S3AuthenticationV4.lowercase(awsHeader.getKey()));
        }
        return signedHeaders.toString();
    }

    private String buildHashedPayload() {
        return AWS_CONTENT_SHA256_HEADER_VALUE;
    }

    private String buildScope() {
        String scope = SCOPE_VALUE;
        scope = scope.replace(DATE_STAMP_NAME, this.getISODate());
        scope = scope.replace(AWS_REGION_NAME, this.region);
        return scope;
    }

    private String buildCanonicalRequestHash(String canonicalRequest) {
        byte[] canonicalRequestHash = S3AuthenticationV4.sha256Hash(canonicalRequest.getBytes());
        return S3AuthenticationV4.hex(canonicalRequestHash);
    }

    private String buildCanonicalRequest(URL url) {
        String canonicalURI = S3AuthenticationV4.uriEncode(url.getPath(), false);
        String canonicalQueryString = this.buildCanonicalQueryString(S3AuthenticationV4.getRequestParametersList(url.getQuery()));
        String canonicalHeaders = this.buildCanonicalHeaders();
        String signedHeaders = this.buildSignedHeaders();
        String hashedPayload = this.buildHashedPayload();
        String canonicalRequest = CANONICAL_REQUEST_VALUE;
        canonicalRequest = canonicalRequest.replace(HTTP_VERB_NAME, this.httpVerb);
        canonicalRequest = canonicalRequest.replace(CANONICAL_URI_NAME, canonicalURI);
        canonicalRequest = canonicalRequest.replace(CANONICAL_QUERY_STRING_NAME, canonicalQueryString);
        canonicalRequest = canonicalRequest.replace(CANONICAL_HEADERS_NAME, canonicalHeaders);
        canonicalRequest = canonicalRequest.replace(SIGNED_HEADERS_NAME, signedHeaders);
        canonicalRequest = canonicalRequest.replace(HASHED_PAYLOAD_NAME, hashedPayload);
        return canonicalRequest;
    }

    private String buildStringToSign(String canonicalRequest) {
        String timeStamp = this.getISODateTime();
        String scope = this.buildScope();
        String canonicalRequestHash = this.buildCanonicalRequestHash(canonicalRequest);
        String stringToSign = STRING_TO_SIGN_VALUE;
        stringToSign = stringToSign.replace(AWS_SIGNATURE_ALGORITHM_NAME, AWS_SIGNATURE_ALGORITHM_VALUE);
        stringToSign = stringToSign.replace(TIMESTAMP_NAME, timeStamp);
        stringToSign = stringToSign.replace(SCOPE_NAME, scope);
        stringToSign = stringToSign.replace(CANONICAL_REQUEST_HASH_NAME, canonicalRequestHash);
        return stringToSign;
    }

    private byte[] buildSigningKey() {
        String secretKey = "AWS4" + this.secretAccessKey;
        byte[] dateKey = S3AuthenticationV4.hmacSHA256Hash(this.getISODate().getBytes(), secretKey.getBytes());
        byte[] regionKey = S3AuthenticationV4.hmacSHA256Hash(this.region.getBytes(), dateKey);
        byte[] serviceKey = S3AuthenticationV4.hmacSHA256Hash(AWS_SERVICE_VALUE.getBytes(), regionKey);
        return S3AuthenticationV4.hmacSHA256Hash(SIGNING_KEY_DATA_VALUE.getBytes(), serviceKey);
    }

    private String buildSignature(URL url) {
        String canonicalRequest = this.buildCanonicalRequest(url);
        String stringToSign = this.buildStringToSign(canonicalRequest);
        byte[] signature = S3AuthenticationV4.hmacSHA256Hash(stringToSign.getBytes(), this.signingKey);
        return S3AuthenticationV4.hex(signature);
    }

    private String getISODate() {
        return ISO_DATE.format(this.creationDate);
    }

    private String getISODateTime() {
        return ISO_DATE_TIME.format(this.creationDate);
    }

    private boolean isExpired() {
        return this.expirationDate == null || this.expirationDate.isBefore(LocalDateTime.now(ZoneOffset.UTC));
    }

    private void ensureValid() {
        if (this.isExpired()) {
            this.creationDate = LocalDateTime.now(ZoneOffset.UTC);
            this.expirationDate = this.creationDate.plusMinutes(14L);
            this.signingKey = this.buildSigningKey();
        }
    }

    String getAuthorizationToken(URL url) {
        if (url == null) {
            throw new NullPointerException("url");
        }
        if (this.accessKeyId == null || this.accessKeyId.isEmpty() || this.secretAccessKey == null || this.secretAccessKey.isEmpty()) {
            return null;
        }
        if (this.lastAuthorizationToken == null || this.isExpired() || !url.toString().contentEquals(this.lastAuthorizedURL.toString())) {
            this.ensureValid();
            this.awsHeaders = this.buildAwsHeaders(url);
            String scope = this.buildScope();
            String signedHeaders = this.buildSignedHeaders();
            String signature = this.buildSignature(url);
            String authorizationToken = AWS_AUTHORIZATION_TOKEN_VALUE;
            authorizationToken = authorizationToken.replace(AWS_SIGNATURE_ALGORITHM_NAME, AWS_SIGNATURE_ALGORITHM_VALUE);
            authorizationToken = authorizationToken.replace(AWS_ACCESS_KEY_ID_NAME, this.accessKeyId);
            authorizationToken = authorizationToken.replace(SCOPE_NAME, scope);
            authorizationToken = authorizationToken.replace(SIGNED_HEADERS_NAME, signedHeaders);
            this.lastAuthorizationToken = authorizationToken = authorizationToken.replace(AWS_SIGNATURE_NAME, signature);
            this.lastAuthorizedURL = url;
        }
        return this.lastAuthorizationToken;
    }

    Map<String, String> getAwsHeaders(URL url) {
        if (this.awsHeaders == null || this.lastAuthorizedURL == null || !url.toString().contentEquals(this.lastAuthorizedURL.toString())) {
            throw new IllegalStateException("Unsigned AWS Headers. Request token before AWS Headers.");
        }
        return this.awsHeaders;
    }

    private static final class PropertyCodePointSorter
    implements Comparator<Property> {
        private PropertyCodePointSorter() {
        }

        @Override
        public int compare(Property p1, Property p2) {
            int i;
            for (i = 0; i < Math.min(p1.getName().length(), p2.getName().length()); ++i) {
                if (p1.getName().codePointAt(i) == p2.getName().codePointAt(i)) continue;
                if (p1.getName().codePointAt(i) < p2.getName().codePointAt(i)) {
                    return -1;
                }
                return 1;
            }
            if (p1.getName().length() != p2.getName().length()) {
                if (p1.getName().length() < p2.getName().length()) {
                    return -1;
                }
                return 1;
            }
            for (i = 0; i < Math.min(p1.getValue().length(), p2.getValue().length()); ++i) {
                if (p1.getValue().codePointAt(i) == p2.getValue().codePointAt(i)) continue;
                if (p1.getValue().codePointAt(i) < p2.getValue().codePointAt(i)) {
                    return -1;
                }
                return 1;
            }
            if (p1.getValue().length() != p2.getValue().length()) {
                if (p1.getValue().length() < p2.getValue().length()) {
                    return -1;
                }
                return 1;
            }
            return 0;
        }
    }
}

